/** 
  Copyright (c) 2010 Freescale Semiconductor
  
  \file  	  Flib.c
  \brief	  This is the Flash Library Main Driver File
  \brief	  Provides functionality to animate graphics
  \author	  Freescale Semiconductor
  \author	  Automotive Systems Solutions Engineering
  \author	  IM, b06623
  \version	  2.0
  \revision	  $Revision: 273 $
  \date  	  $Date: 2011-11-16 15:42:33 -0600 (Wed, 16 Nov 2011) $  
  
* History:  5/Jan/2009 - Initial Version
*           11/May/2009 - Last changes applied
*			11/Ago/2010	- Added Size animation code functionality. Context Switching
			1/Sept/2010	- Fixed scale animation bugs related to front and back buffer, context switching and combination with
						  position animation.
			30/Sept/2010 - FLIB_AnimCancel added.
			6/Oct/2010 - Updated custom animation: functionality OK
			10/dec/2011- Updated, for the case of NULL_PTR animations, layer data is used.  			  

*	MISRA VIOLATIONS:
	- [ MISRA 11.2 ]
	- [ MISRA 16.9 ]

* Copyright (c) 2010, Freescale, Inc.  All rights reserved.
*
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
  
*/

#include "Flib.h"
#include "Display.h"
#include "FlibInternal.h"
#include "..\SERVICES\DMAHandler.h"
#include "..\SERVICES\AnimationCtrl.h"
#include "VarAnimator.h"


/* Driver Variables */
uint8_t FLIB_Ctx = 0;

#ifdef DCULITE
#define FLIB_CTXNO	2
#else
#define FLIB_CTXNO	1
#endif

FLIB_State_t FLIB_State[FLIB_CTXNO] = 
{
	FLIB_STATE_UNINIT,
#ifdef DCULITE	
	FLIB_STATE_UNINIT
#endif	
};

FLIB_CallbackType FLIB_CallbackAnimator[FLIB_CTXNO];

FLIB_AnimLayer_t FLIB_AnimationData_1[FLIB_MAXANICTX0];
#ifdef DCULITE
FLIB_AnimLayer_t FLIB_AnimationData_2[FLIB_MAXANICTX1];
#endif
FLIB_AnimLayer_t* FLIB_AnimationData = FLIB_AnimationData_1;

Graphics_Object_t *FLIB_GObjectCurrent[FLIB_CTXNO];
Graphics_Object_t FLIB_GObjectTmpBuffer1[FLIB_CTXNO];
Graphics_Object_t FLIB_GObjectTmpBuffer2[FLIB_CTXNO]; 
Graphics_Object_t FLIB_GObjectTmpBuffer0[FLIB_CTXNO];

uint32_t FLIB_FrameCount[FLIB_CTXNO];
uint32_t FLIB_FrameCurrent[FLIB_CTXNO];

FLIB_CustomAni_t FLIB_CustomDataStack[FLIB_CTXNO][FLIB_MAXANICUSTOM];

static void FLIB_FrameCountSync(void);
#ifdef DCULITE
static void FLIB_FrameCountSync2(void);
#endif

static void FLIB_Interpolate(FLIB_Interpolation_t transition, int32_t *x, int32_t *dx, int32_t xf, int32_t t, int32_t duration);
static uint32_t FLIB_InterpolateSimple(int32_t xi, int32_t xf, int32_t t, int32_t duration);
static void FLIB_ObjectYOffset(uint32_t *address, uint8_t BPP, uint32_t Yoffset, uint32_t Width);
static void FLIB_Pixel2Bytes(uint8_t BPP, uint32_t *size);
static uint16_t FLIB_CropWidth(uint8_t BPP, uint16_t Width);
static void FLIB_CustomAnimationManager(void);


/**
* \brief	FLIB_SetContextXXX - This function sets the DCU context: DCU or DCULITE
* \brief	Should be called before doing any initialization or anything, depending the DCU used.
* \author	IM, b06623
* \param	void
* \return	void
* \todo
*/
void FLIB_SetContextDCU(void) 
{
	FLIB_Ctx = 0;
	GALLO_SetContextDCU();
	Display_SelectDCU();
	MCManager_SetContext(0);
	FLIB_AnimationData = FLIB_AnimationData_1;	
}

#ifdef DCULITE	
void FLIB_SetContextDCULITE(void) 
{
	FLIB_Ctx = 1;
	GALLO_SetContextDCULITE();
	Display_SelectDCULITE();
	MCManager_SetContext(1);
	FLIB_AnimationData = FLIB_AnimationData_2;			
}
#endif

/**
* \brief	FLIB_Init - This function initializes the internal state of
* \brief	the FlashLib. It therefore takes care for all memory allocations
* \brief	and initializations of variables and underlying libraries
* \author	IM, b06623
* \param	void
* \return	void
* \todo
*/
void FLIB_Init(void) 
{
    uint16_t i;
    uint8_t	maxAni;
    
    /*Initialization Data*/
    FLIB_Ctx = 0;
    FLIB_State[FLIB_Ctx] = FLIB_STATE_UNINIT;
    FLIB_AnimationData = FLIB_AnimationData_1;
    
    FLIB_State[FLIB_Ctx] = FLIB_STATE_OK;
    /* VIOLATION to [MISRA 11.2] [MISRA 16.9] Rationale: NULL pointers used to identify empty objects */ 
    FLIB_CallbackAnimator[FLIB_Ctx] = NULL_PTR;   
    FLIB_FrameCount[FLIB_Ctx] = 1u;
    FLIB_FrameCurrent[FLIB_Ctx] = 1u;  
    
    if(FLIB_Ctx == 0)
    {
    	maxAni = FLIB_MAXANICTX0;
    	/* VIOLATION to [MISRA 16.9] Rationale: IS the only way to set a callback */       
    	DCU_SetCallbackProgDone(FLIB_FrameCountSync);	  	
    }
#ifdef DCULITE    
    else
    {
    	maxAni = FLIB_MAXANICTX1;	
    	/* VIOLATION to [MISRA 16.9] Rationale: IS the only way to set a callback */       
    	DCU_SetCallbackProgDone(FLIB_FrameCountSync2);  
    	   	
    }
    DMA_SetDCUContextCallback(FLIB_SetContextDCU);
    DMA_SetDCULITEContextCallback(FLIB_SetContextDCULITE);    
#endif    
    for(i = 0u; i < maxAni; i++)
    {
		FLIB_AnimationData[i].duration = 0u;
    }
    
    for(i=0; i<FLIB_MAXANICUSTOM; i++)
	{
		FLIB_CustomDataStack[FLIB_Ctx][i].callback = NULL_PTR;
	}     
}

/**
* \brief	FLIB_DeInit - De-initialize the library and free all allocated
* \brief	memory. The function also de-initializes the underlying libraries
* \brief	if required
* \author	IM, b06623
* \param	void
* \return	void
* \todo
*/
void FLIB_DeInit(void)
{
    uint16_t i;
    uint8_t	maxAni;
    if(FLIB_Ctx == 0)
    {
    	maxAni = FLIB_MAXANICTX0;
    }
    else
    {
    	maxAni = FLIB_MAXANICTX1;	
    }
    for(i = FLIB_MEMSTARTNUMBER; i < (FLIB_MEMSTARTNUMBER + maxAni); i++)
    {
		GALLO_DeAllocate((uint8_t)i);
    }
    FLIB_State[FLIB_Ctx] = FLIB_STATE_UNINIT;
    FLIB_CallbackAnimator[FLIB_Ctx] = NULL_PTR; 
}

/* cancels a running animation */
void FLIB_AnimCancel(uint8_t Layer)
{
	FLIB_AnimationData[Layer].duration = 0;	
}

/**
* \brief	FLIB_Setup - Sets up the animator by registering the callback
* \brief	function actually defining the animation sequence if required.
* \author	IM, b06623
* \param	Callback: Function pointer to the animation setup function
* \param	provided by the application
* \return	void
* \todo
*/
void FLIB_Setup(FLIB_CallbackType Callback)
{
    /* VIOLATION to [MISRA 16.9] Rationale: IS the only way to set a callback */ 
    FLIB_CallbackAnimator[FLIB_Ctx] = Callback;   
}

uint8_t FLIB_ScaleChannel[FLIB_CTXNO];
uint8_t FLIB_ScaleLayer[FLIB_CTXNO];
uint8_t FLIB_ScaleDone[FLIB_CTXNO];

 

void FLIB_InternalUpdateScaleRegs(void)
{
	DCU_LayerAddress(FLIB_ScaleLayer[FLIB_Ctx]) = FLIB_GObjectTmpBuffer2[FLIB_Ctx].address;
	DCU_LayerSetWidth(FLIB_ScaleLayer[FLIB_Ctx],FLIB_GObjectTmpBuffer2[FLIB_Ctx].width);
	DCU_LayerSetHeight(FLIB_ScaleLayer[FLIB_Ctx],FLIB_GObjectTmpBuffer2[FLIB_Ctx].height);
	FLIB_ScaleDone[FLIB_Ctx] = 1;
}

void FLIB_InternalScalePaste(void)
{   
    Graphics_Paste(FLIB_ScaleChannel[FLIB_Ctx], &FLIB_GObjectTmpBuffer1[FLIB_Ctx], &FLIB_GObjectTmpBuffer2[FLIB_Ctx],0,0,FLIB_InternalUpdateScaleRegs);
    DMA_BWC(FLIB_ScaleChannel[FLIB_Ctx]) = 2;
    DMA_Start(FLIB_ScaleChannel[FLIB_Ctx]);
}

void FLIB_InternalScaleBlank(void)
{
    /* Add blanking  before pasting */
    Graphics_BlankArea(FLIB_ScaleChannel[FLIB_Ctx], &FLIB_GObjectTmpBuffer2[FLIB_Ctx], 0x00000000, FLIB_InternalScalePaste);
    DMA_BWC(FLIB_ScaleChannel[FLIB_Ctx]) = 2;
    DMA_Start(FLIB_ScaleChannel[FLIB_Ctx]);
}

/**
* \brief	FLIB_RenderNextFrame - Render the next frame frame. Prepares
* \brief	the required images, sets up the DCU to display the frame. 
* \author	IM, b06623
* \param	void
* \return	FLIB_Error_t: GRAPHICS_ERROR_OK, GRAPHICS_ERROR_NO_CALLBACK.
* \todo
*/
FLIB_Error_t FLIB_RenderNextFrame(void)
{
    /* Auxiliar Variables for calculations */
    int32_t 	s32temp0;
    int32_t 	s32temp1;
    uint32_t 	u32temp3;
    uint16_t 	u16temp4;
    uint16_t 	u16temp6;
    uint8_t  	u8temp5;       
    /* Variable i, is used as layer iterator */
    uint16_t 	i;
    /* Flag to control the reload of the values in the DCU Layer*/
    uint16_t 	reload;
    /* Scale Flags. 0 means normal, 1 means cancel, 2 means stay */
    uint16_t 	scaleFlag;
    /* Defines the max number of animations */
    uint8_t		maxAni;     
    /* Check the error in this method */
    FLIB_Error_t	error;   
    
    error = FLIB_ERROR_OK;
    
    if(FLIB_State[FLIB_Ctx] == FLIB_STATE_UNINIT)
    {
    	return FLIB_ERROR_UNINIT;
    }
    
    /* Select Context to work in (DCU or DCULITE) */
    if(FLIB_Ctx == 0)
    {
    	maxAni = FLIB_MAXANICTX0;
    	//AnimatorF();
    	
    }
    else
    {
    	maxAni = FLIB_MAXANICTX1;	
    }    
    
    /* VIOLATION to [MISRA 16.9] Rationale: IS the only way to verify a callback is not null */ 
    if(FLIB_CallbackAnimator[FLIB_Ctx] == NULL_PTR)
    {
		error = FLIB_ERROR_NO_CALLBACK;
    }
    else
    {
		FLIB_CallbackAnimator[FLIB_Ctx](FLIB_FrameCurrent[FLIB_Ctx]);
    }            
    for(i=0u; i < maxAni; i++)
    {
	if(FLIB_AnimationData[i].duration != 0u)
	{
	    reload = 0;
	    scaleFlag = 2;
	    
	    /* ************************************************************** */
	    /* ********************* INITLIZE IMAGE SECTION ***************** */
	    /* ************************************************************** */
	    if( FLIB_AnimationData[i].currentFrame == 0u)
	    {
			reload = 1u;
			scaleFlag = 0u;
			if(FLIB_AnimationData[i].gobject.address == (uint32_t)NULL_PTR)
			{
				FLIB_CopyGraphicObjectLayer((uint8_t)i, &FLIB_AnimationData[i].gobject);
			}
	    }
	    FLIB_GObjectCurrent[FLIB_Ctx] = &FLIB_AnimationData[i].gobject;
	    
	    /* ************************************************************** */
	    /* **************** ALPHA CHNL ANIMATION SECTION **************** */
	    /* ************************************************************** */
	    if(FLIB_AnimationData[i].anim.alpha !=NULL_PTR)
	    {
			s32temp0 = FLIB_AnimationData[i].state.alpha/65536;
			DCU_LayerAlpha( i ) = ((uint8_t)s32temp0);
					
			if(FLIB_AnimationData[i].anim.alpha->transition == FLIB_TABLE)
			{
			    s32temp0 = (int32_t)FLIB_AnimationData[i].anim.alpha->inter_Table[FLIB_AnimationData[i].currentFrame];
			    s32temp0 = s32temp0 * 65536;
			    FLIB_AnimationData[i].state.alpha = s32temp0;
			}
			else
			{
			    FLIB_Interpolate(FLIB_AnimationData[i].anim.alpha->transition, &FLIB_AnimationData[i].state.alpha,
				    &FLIB_AnimationData[i].state.dalpha, FLIB_AnimationData[i].anim.alpha->end,
				    (int32_t)FLIB_AnimationData[i].currentFrame, (int32_t)FLIB_AnimationData[i].duration);
			}		
	    }
	    
	    /* ************************************************************** */
	    /* ***************** POSITION ANIMATION SECTION ***************** */
	    /* ************************************************************** */	    
	    if(FLIB_AnimationData[i].anim.pos !=NULL_PTR)
	    {		
		if(FLIB_AnimationData[i].state.y < 0)
		{		    
		    DCU_LayerSetY(i, 0u);
		    s32temp1 = (-1 * FLIB_AnimationData[i].state.y)/65536;
		    u16temp4 = (uint16_t)s32temp1;
		}
		else
		{
		    u16temp4 = 0u;
		    DCU_LayerSetY(i, (FLIB_AnimationData[i].state.y)/65536);	
		}

		s32temp0 = FLIB_AnimationData[i].state.y/65536;
		s32temp0 = s32temp0 + ((int32_t)FLIB_GObjectCurrent[FLIB_Ctx]->height);

		/* Set an offset on the starting address */
		u32temp3 = FLIB_GObjectCurrent[FLIB_Ctx]->address;
		FLIB_ObjectYOffset(&u32temp3, FLIB_GObjectCurrent[FLIB_Ctx]->BPP, u16temp4, FLIB_GObjectCurrent[FLIB_Ctx]->width);
		DCU_LayerAddress(i) = u32temp3;

		DCU_LayerSetWidth( i,  FLIB_GObjectCurrent[FLIB_Ctx]->width);
		s32temp1 = FLIB_AnimationData[i].state.x/65536;
		DCU_LayerSetX( i, ((uint16_t)s32temp1));

		/* Check if the object is out of bounds */
		/* Because Spectrum cut2, is not needed anymore */
		/*
		s32temp1 = (int32_t)DCU_GetMaxY();
		if(s32temp0 >= s32temp1)
		{
		    u16temp4 = (DCU_GetMaxY() - DCU_LayerGetY(i));
		}
		else
		{
		    u16temp4 = (FLIB_GObjectCurrent[FLIB_Ctx]->height - u16temp4);
		}
		*/
		if(u16temp4 >= FLIB_GObjectCurrent[FLIB_Ctx]->height)
		{
			u16temp4 = 0;	
		}
		else
		{
			u16temp4 = (uint16_t)(FLIB_GObjectCurrent[FLIB_Ctx]->height - u16temp4);
		}		
		DCU_LayerSetHeight( i, u16temp4);


		if(FLIB_AnimationData[i].anim.pos->transition == FLIB_TABLE)
		{
		    s32temp0 = (int32_t)FLIB_AnimationData[i].anim.pos->inter_Table[FLIB_AnimationData[i].currentFrame].x;
		    s32temp0 = s32temp0 * 65536;
		    FLIB_AnimationData[i].state.x = s32temp0;
		    s32temp0 = (int32_t)FLIB_AnimationData[i].anim.pos->inter_Table[FLIB_AnimationData[i].currentFrame].y;
		    s32temp0 = s32temp0 * 65536;
		    FLIB_AnimationData[i].state.y = s32temp0;
		}
		else
		{
		    FLIB_Interpolate(FLIB_AnimationData[i].anim.pos->transition, &FLIB_AnimationData[i].state.x,
			    &FLIB_AnimationData[i].state.dx, FLIB_AnimationData[i].anim.pos->end.x,
			    (int32_t)FLIB_AnimationData[i].currentFrame, (int32_t)FLIB_AnimationData[i].duration);
		    FLIB_Interpolate(FLIB_AnimationData[i].anim.pos->transition, &FLIB_AnimationData[i].state.y,
			    &FLIB_AnimationData[i].state.dy, FLIB_AnimationData[i].anim.pos->end.y,
			    (int32_t)FLIB_AnimationData[i].currentFrame, (int32_t)FLIB_AnimationData[i].duration);
		}

	    }
	    /* ************************************************************** */
	    /* ******************* INITIALIZE IMAGE SECTION ***************** */
	    /* ************************************************************** */
	    if(reload == 1)
	    {
		
			DCU_LayerAddress( i ) = FLIB_GObjectCurrent[FLIB_Ctx]->address;
		
			if(FLIB_AnimationData[i].anim.pos == NULL_PTR && FLIB_AnimationData[i].anim.size == NULL_PTR)
			{
		    	DCU_LayerSetY( i, FLIB_GObjectCurrent[FLIB_Ctx]->y );		       
		    	DCU_LayerSetHeight( i, FLIB_GObjectCurrent[FLIB_Ctx]->height);
		    	DCU_LayerSetWidth( i,  FLIB_GObjectCurrent[FLIB_Ctx]->width);
		    	DCU_LayerSetX( i, FLIB_GObjectCurrent[FLIB_Ctx]->x );
			}

			DCU_LayerOffset( i ) = FLIB_GObjectCurrent[FLIB_Ctx]->CLUTOffset;
			DCU_LayerBPP( i ) = FLIB_GObjectCurrent[FLIB_Ctx]->BPP;

			DCU_SetChroma( (DCU_Plane_t)i, FLIB_GObjectCurrent[FLIB_Ctx]->initialChromaMax, FLIB_GObjectCurrent[FLIB_Ctx]->initialChromaMin);

			if(FLIB_AnimationData[i].anim.alpha == NULL_PTR)
			{
			    DCU_LayerAlpha( i ) = FLIB_GObjectCurrent[FLIB_Ctx]->initialAlpha;
			}
			
			DCU_TileMode(i, 0, 0, 0);


			if(FLIB_GObjectCurrent[FLIB_Ctx]->coding==GRAPHICS_CODING_RLEINT)
			{
				DCU_RLE(i,1,Graphics_GetCompSize(FLIB_GObjectCurrent[FLIB_Ctx]->ext));	
			}		
			else		

	    	if(FLIB_GObjectCurrent[FLIB_Ctx]->coding==GRAPHICS_CODING_TILE)
	    	{
	    		DCU_TileMode(i, 1, Graphics_GetTileW(FLIB_GObjectCurrent[FLIB_Ctx]->ext), Graphics_GetTileH(FLIB_GObjectCurrent[FLIB_Ctx]->ext));
	    	}

			DCU_LayerEnable(i);		
		}	    

	    
	    FLIB_AnimationData[i].currentFrame++;
	    
	    if(FLIB_AnimationData[i].duration == FLIB_AnimationData[i].currentFrame)
	    {
			FLIB_AnimationData[i].duration = 0u;
	    }
	}
	
    }
	
	/* ************************************************************** */
	/* ******************* CUSTOM ANIMATION SECTION ***************** */
	/* ************************************************************** */
	FLIB_CustomAnimationManager();
	
    FLIB_FrameCurrent[FLIB_Ctx]++;	
    return error;
}

/**
* \brief	FLIB_GetState - Check if there is a pending update not yet
* \brief	displayed by the DCU. This function can be used to ensure that
* \brief	each frame is displayed at least once.
* \author	IM, b06623
* \param	void
* \return	FLIB_State_t: FLIB_STATE_OK, FLIB_STATE_UPDATE_PENDING.
* \todo
*/
FLIB_State_t FLIB_GetState(void)
{
    FLIB_State_t st;
    
    if(FLIB_FrameCurrent[FLIB_Ctx] == FLIB_FrameCount[FLIB_Ctx])
    {
		st = FLIB_STATE_OK;
    }
    else
    {
		st = FLIB_STATE_UPDATE_PENDING;	
    }
    return st;
}

void FLIB_Sync(void)
{
	FLIB_FrameCount[FLIB_Ctx] = FLIB_FrameCurrent[FLIB_Ctx];
}

/**
* \brief	FLIB_AnimSetupCustom - Register a function providing the actual
* \brief	position, alpha value, color, scaling, scrolling and object for
* \brief	each frame number. This callback is then called by the animator.
* \author	IM, b06623
* \param	sprite: Graphic Object where the animation is performed.
* \param	duration: Number of frames the animation takes. 
* \param	Callback: pointer to callback animation function
* \return	FLIB_Error_t: FLIB_ERROR_OK (service was accepted),
* \return	FLIB_ERROR_ANIM_POS (Error in alpha animation parameters)
* \todo
*/
FLIB_Error_t FLIB_AnimSetupCustom
(
	uint32_t	duration, 			//	cant be zero. FLIB_FOREVER for eternal duration
	uint8_t		rate,				//  repetition rate
	uint8_t		offset,				//	offset, based on the rate
	FLIB_CallbackType callback		//	callback, arguments contain relative frame, duration...
)
{
	uint32_t i;
	
	if(callback == NULL_PTR)
	{
		return FLIB_ERROR_NO_CALLBACK;
	}	
	if(rate== 0)
	{
		return FLIB_ERROR_PARAM;
	}
	if(duration == 0)
	{
		return FLIB_ERROR_PARAM;
	}
	if(offset >= rate)
	{
		return FLIB_ERROR_PARAM;
	}	
	for(i=0; i<FLIB_MAXANICUSTOM; i++)
	{
		if(FLIB_CustomDataStack[FLIB_Ctx][i].callback == NULL_PTR)
		{
			break;
		}
	}
	
	if(i == FLIB_MAXANICUSTOM)
	{
		return FLIB_ERROR_FULL;
	}
			
	FLIB_CustomDataStack[FLIB_Ctx][i].rate = rate;
	FLIB_CustomDataStack[FLIB_Ctx][i].duration = duration;
	FLIB_CustomDataStack[FLIB_Ctx][i].offset = offset;
	FLIB_CustomDataStack[FLIB_Ctx][i].callback = callback;
	FLIB_CustomDataStack[FLIB_Ctx][i].frameLatch = FLIB_FrameCurrent[FLIB_Ctx];
}

FLIB_Error_t FLIB_KillCustomAni(FLIB_CallbackType callback)
{
	uint32_t i;
	
	for(i=0; i<FLIB_MAXANICUSTOM; i++)
	{
		if(FLIB_CustomDataStack[FLIB_Ctx][i].callback == callback)
		{
			FLIB_CustomDataStack[FLIB_Ctx][i].callback = NULL_PTR;
			break;
		}
	}	
}

static void FLIB_CustomAnimationManager(void)
{
	uint32_t i;
	uint32_t relative_fr;
	uint32_t slot;
	uint32_t zero;	
	
	for(i=0; i<FLIB_MAXANICUSTOM; i++)
	{
		if(FLIB_CustomDataStack[FLIB_Ctx][i].callback != NULL_PTR)
		{
			/* calculate current slot */
			slot = FLIB_FrameCurrent[FLIB_Ctx]%(FLIB_CustomDataStack[FLIB_Ctx][i].rate);
			/* calculate relative frame based on anchor */
			relative_fr = (FLIB_FrameCurrent[FLIB_Ctx] - FLIB_CustomDataStack[FLIB_Ctx][i].frameLatch)/FLIB_CustomDataStack[FLIB_Ctx][i].rate;
			/* check if current slot corresponds to this custom animation */
			zero = (slot + FLIB_CustomDataStack[FLIB_Ctx][i].rate - FLIB_CustomDataStack[FLIB_Ctx][i].offset)%
					FLIB_CustomDataStack[FLIB_Ctx][i].rate;
			if(zero == 0)
			{
				FLIB_CustomDataStack[FLIB_Ctx][i].callback(relative_fr);
				
				/* check duration of the callback and kill it after a while */
				if(FLIB_CustomDataStack[FLIB_Ctx][i].duration == relative_fr)
				{
					if(FLIB_CustomDataStack[FLIB_Ctx][i].duration != FLIB_FOREVER)
					{
						FLIB_CustomDataStack[FLIB_Ctx][i].callback = NULL_PTR;	
					}						
				}
			}				
		}
	}
}

/**
* \brief	Display_CopyGraphicObjectLayer - This function copies all the data from one Layer to an G-Object.
* \brief	There are some parameters that cannot be obtained from a layer, such as: Clut Address and Clut size
* \author	IM, b06623
* \param	DCU_Plane_t Layer: Is the source layer, where data is obtained to be copied.
* \param    Graphics_Object_t* target: Is the target object.
* \return	void
* \todo
*/
void FLIB_CopyGraphicObjectLayer(uint8_t Layer, Graphics_Object_t *target)
{
    target->address = DCU_LayerAddress(Layer);
    target->width = (uint16_t)DCU_LayerGetWidth(Layer);
    target->height = (uint16_t)DCU_LayerGetHeight(Layer);
    /* VIOLATION to [MISRA 11.2] Rationale: NULL pointers used to identify empty objects */
    target->CLUT = NULL_PTR;
    target->CLUTsize = 0u;
    target->CLUTOffset = (uint16_t)DCU_LayerOffset(Layer);
    /* uint8_t used only for storage */ 
    target->BPP = (uint8_t)DCU_LayerBPP(Layer);
    /* uint8_t used only for storage */ 
    target->initialAlpha = (uint8_t)DCU_LayerAlpha(Layer);
    target->initialChromaMax = ((uint32_t)DCUCTX.LAYER[ Layer ].CTRLDESCL5.B.CKMAX_R<<16) +
    						   ((uint32_t)DCUCTX.LAYER[ Layer ].CTRLDESCL5.B.CKMAX_G<<8) +
    						   ((uint32_t)DCUCTX.LAYER[ Layer ].CTRLDESCL5.B.CKMAX_B);	
    target->initialChromaMin = ((uint32_t)DCUCTX.LAYER[ Layer ].CTRLDESCL6.B.CKMIN_R<<16) +
    						   ((uint32_t)DCUCTX.LAYER[ Layer ].CTRLDESCL6.B.CKMIN_G<<8) +
    						   ((uint32_t)DCUCTX.LAYER[ Layer ].CTRLDESCL6.B.CKMIN_B);
     
    target->coding = GRAPHICS_CODING_RAWINT;
    target->x = (int16_t)DCU_LayerGetX(Layer);
    target->y = (int16_t)DCU_LayerGetY(Layer); 
    target->ext = NULL_PTR;
}


/* ************************************************************** */
/* ***************** PRIVATE STATIC HELP FUNCTIONS ************** */
/* ************************************************************** */
 
static void FLIB_FrameCountSync(void)
{
    FLIB_FrameCount[0]++;
}

#ifdef DCULITE
static void FLIB_FrameCountSync2(void)
{
    FLIB_FrameCount[1]++;

}
#endif

 
static void FLIB_Interpolate(FLIB_Interpolation_t transition, int32_t *x, int32_t *dx, int32_t xf, int32_t t, int32_t duration)
{
    int32_t tmp;
    
    if(transition == FLIB_LINEAR)
    {
	/* Improved possitioning, in final frame pos = end */
	if(t >= (duration - 2))
	{
	    *x = xf*65536;
	}
	else
	{
	    *x += *dx;
	}
    }
    else if(transition == FLIB_DYNAMIC)
    {		
	/* Calculate inflection point (33% = 1/3) */
	tmp = duration/3;

	/* Ramp up */
	if(t < tmp)
	{
	    *x += ((t + 1)*(*dx))/tmp;
	}
	else if(t < (duration - tmp))
	{
	    *x += *dx;
	}
	else
	{
	    if(t == (duration - tmp))
	    {
		/* Re-calculate delta distance */
		*dx = (xf*65536) - *x;
		*dx = (2*(*dx))/tmp;
	    }
	    /* Improved possitioning, in final frame pos = end */
	    if(t >= (duration - 2) )
	    {
		*x = xf * 65536;
	    }
	    else
	    {
		*x += ((duration - t - 1)*(*dx))/tmp;
	    }
	}
    }
    else
    {
    }    
}

static uint32_t FLIB_InterpolateSimple(int32_t xi, int32_t xf, int32_t t, int32_t duration)
{
    xf = (t*(xf - xi))/(duration - 1);
    xf = xi + xf;
    return ((uint32_t)xf);
}

static void FLIB_ObjectYOffset(uint32_t *address, uint8_t BPP, uint32_t Yoffset, uint32_t Width)
{
    switch(BPP)
    {
	case GRAPHICS_1BPP:
	    *address = *address + ((Yoffset*Width)/8u);
	    break;
	case GRAPHICS_2BPP:
	    *address = *address + ((Yoffset*Width)/4u);
	    break;
	case GRAPHICS_4BPP:
	case GRAPHICS_T4BPP:
	case GRAPHICS_L4BPP:
	    *address = *address + ((Yoffset*Width)/2u);
	    break;
	case GRAPHICS_8BPP:
	case GRAPHICS_T8BPP:
	case GRAPHICS_L8BPP:
	    *address = *address + (Yoffset*Width);
	    break;
	case GRAPHICS_565RGB:
	case GRAPHICS_1555ARGB:
	case GRAPHICS_4444ARGB:
#ifdef DCU3
	case GRAPHICS_APAL8:
	case GRAPHICS_YCBCR422:
#endif	
	    *address = *address + ((Yoffset*Width)*2u);
	    break;
	case GRAPHICS_24BPP:
	    *address = *address + ((Yoffset*Width)*3u);
	    break;
	case GRAPHICS_32BPP:	
	    *address = *address + ((Yoffset*Width)*4u);
	    break;	    
	default:
	    break;
    }
}

static uint16_t FLIB_CropWidth(uint8_t BPP, uint16_t Width)
{	
    switch(BPP)
    {
	case GRAPHICS_1BPP:
	    Width = (uint16_t)((Width>>5)<<5);
	    break;
	case GRAPHICS_2BPP:
	    Width = (uint16_t)((Width>>4)<<4);
	    break;
	case GRAPHICS_4BPP:
	case GRAPHICS_T4BPP:
	case GRAPHICS_L4BPP:
	    Width = (uint16_t)((Width>>3)<<3);
	    break;
	case GRAPHICS_8BPP:
	case GRAPHICS_T8BPP:
	case GRAPHICS_L8BPP:
	case GRAPHICS_24BPP:
#ifdef DCU3
	case GRAPHICS_YCBCR422:
#endif	
	    Width = (uint16_t)((Width>>2)<<2);
	    break;
	case GRAPHICS_565RGB:
	case GRAPHICS_1555ARGB:
	case GRAPHICS_4444ARGB:
#ifdef DCU3
	case GRAPHICS_APAL8:
#endif	
	    Width = (uint16_t)((Width>>1)<<1);
	    break;	    	    
	default:
	    break;
    }
    
    return Width;
}

static void FLIB_Pixel2Bytes(uint8_t BPP, uint32_t *size)
{
    switch(BPP)
    {
	case GRAPHICS_1BPP:
	    *size = *size/8u;
	    break;
	case GRAPHICS_2BPP:
	    *size = *size/4u;
	    break;
	case GRAPHICS_4BPP:
	case GRAPHICS_T4BPP:
	case GRAPHICS_L4BPP:
	    *size = *size/2u;
	    break;
	case GRAPHICS_565RGB:
	case GRAPHICS_1555ARGB:
	case GRAPHICS_4444ARGB:
#ifdef DCU3
	case GRAPHICS_APAL8:
	case GRAPHICS_YCBCR422:
#endif	
	    *size = *size*2u;
	    break;
	case GRAPHICS_24BPP:
	    *size = *size*3u;
	    break;
	case GRAPHICS_32BPP:		
	    *size = *size*4u;
	    break;
	case GRAPHICS_8BPP:
	case GRAPHICS_T8BPP:
	case GRAPHICS_L8BPP:	    
	default:
	    /* Do nothing */
	    break;
    }
}

